/*
 * Licensed to the OpenAirInterface (OAI) Software Alliance under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The OpenAirInterface Software Alliance licenses this file to You under
 * the OAI Public License, Version 1.0  (the "License"); you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.openairinterface.org/?page_id=698
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *-------------------------------------------------------------------------------
 * For more information about the OpenAirInterface (OAI) Software Alliance:
 *      contact@openairinterface.org
 */

/*! \file rar_tools.c
 * \brief random access tools
 * \author Raymond Knopp and navid nikaein
 * \date 2011 - 2014
 * \version 1.0
 * @ingroup _mac

 */

#include "defs.h"
#include "proto.h"
#include "extern.h"
#include "SIMULATION/TOOLS/defs.h"
#include "UTIL/LOG/log.h"
#include "OCG.h"
#include "OCG_extern.h"
#include "UTIL/OPT/opt.h"
#include "common/ran_context.h"

#define DEBUG_RAR

extern unsigned int  localRIV2alloc_LUT25[512];
extern unsigned int  distRIV2alloc_LUT25[512];
extern unsigned short RIV2nb_rb_LUT25[512];
extern unsigned short RIV2first_rb_LUT25[512];
extern RAN_CONTEXT_t RC;

//------------------------------------------------------------------------------
unsigned short fill_rar(
  const module_id_t module_idP,
  const int         CC_id,
  const frame_t     frameP,
  uint8_t*    const dlsch_buffer,
  const uint16_t    N_RB_UL,
  const uint8_t input_buffer_length
)
//------------------------------------------------------------------------------
{

  RA_HEADER_RAPID *rarh = (RA_HEADER_RAPID *)dlsch_buffer;
  //  RAR_PDU *rar = (RAR_PDU *)(dlsch_buffer+1);
  uint8_t *rar = (uint8_t *)(dlsch_buffer+1);
  int i,ra_idx = -1;
  RA_TEMPLATE *RA_template;

  AssertFatal(CC_id < MAX_NUM_CCs, "CC_id %u < MAX_NUM_CCs %u", CC_id, MAX_NUM_CCs);

  for (i=0; i<NB_RA_PROC_MAX; i++) {
    if (RC.mac[module_idP]->common_channels[CC_id].RA_template[i].generate_rar == 1) {
      ra_idx=i;
      RC.mac[module_idP]->common_channels[CC_id].RA_template[i].generate_rar = 0;
      break;
    }
  }
  RA_template = &RC.mac[module_idP]->common_channels[CC_id].RA_template[ra_idx];
 
  //DevAssert( ra_idx != -1 );
  if (ra_idx==-1)
    return(0);

  // subheader fixed
  rarh->E                     = 0; // First and last RAR
  rarh->T                     = 1; // 0 for E/T/R/R/BI subheader, 1 for E/T/RAPID subheader
  rarh->RAPID                 = RA_template->preamble_index; // Respond to Preamble 0 only for the moment
  rar[4] = (uint8_t)(RA_template->rnti>>8);
  rar[5] = (uint8_t)(RA_template->rnti&0xff);
  //RA_template->timing_offset = 0;
  RA_template->timing_offset /= 16; //T_A = N_TA/16, where N_TA should be on a 30.72Msps
  rar[0] = (uint8_t)(RA_template->timing_offset>>(2+4)); // 7 MSBs of timing advance + divide by 4
  rar[1] = (uint8_t)(RA_template->timing_offset<<(4-2))&0xf0; // 4 LSBs of timing advance + divide by 4
  RA_template->msg3_first_rb=1;
  RA_template->msg3_nb_rb=1;
  uint16_t rballoc = mac_computeRIV(N_RB_UL,RA_template->msg3_first_rb,RA_template->msg3_nb_rb); // first PRB only for UL Grant
  rar[1] |= (rballoc>>7)&7; // Hopping = 0 (bit 3), 3 MSBs of rballoc
  rar[2] = ((uint8_t)(rballoc&0xff))<<1; // 7 LSBs of rballoc
  RA_template->msg3_mcs = 10;
  RA_template->msg3_TPC = 3;
  RA_template->msg3_ULdelay = 0;
  RA_template->msg3_cqireq = 0;
  rar[2] |= ((RA_template->msg3_mcs&0x8)>>3);  // mcs 10
  rar[3] = (((RA_template->msg3_mcs&0x7)<<5)) | ((RA_template->msg3_TPC&7)<<2) | ((RA_template->msg3_ULdelay&1)<<1) | (RA_template->msg3_cqireq&1);

  LOG_D(MAC,"[eNB %d][RAPROC] CC_id %d Frame %d Generating RAR (%02x|%02x.%02x.%02x.%02x.%02x.%02x) for ra_idx %d, CRNTI %x,preamble %d/%d,TIMING OFFSET %d\n",
        module_idP, CC_id,
        frameP,
        *(uint8_t*)rarh,rar[0],rar[1],rar[2],rar[3],rar[4],rar[5],
        ra_idx,
        RA_template->rnti,
        rarh->RAPID,RC.mac[module_idP]->common_channels[CC_id].RA_template[0].preamble_index,
        RA_template->timing_offset);

  if (opt_enabled) {
    trace_pdu(1, dlsch_buffer, input_buffer_length, module_idP, 2, 1,
        RC.mac[module_idP]->frame, RC.mac[module_idP]->subframe, 0, 0);
    LOG_D(OPT,"[eNB %d][RAPROC] CC_id %d RAR Frame %d trace pdu for rnti %x and  rapid %d size %d\n",
          module_idP, CC_id, frameP, RA_template->rnti,
          rarh->RAPID, input_buffer_length);
  }

  return(RA_template->rnti);
}

#ifdef Rel14
//------------------------------------------------------------------------------
unsigned short fill_rar_br(eNB_MAC_INST *eNB,
			   int CC_id,
			   RA_TEMPLATE        *RA_template,      
			   const frame_t      frameP,
			   const sub_frame_t  subframeP,
			   uint8_t*    const  dlsch_buffer,
			   const uint8_t      ce_level
)
//------------------------------------------------------------------------------
{

  RA_HEADER_RAPID *rarh = (RA_HEADER_RAPID *)dlsch_buffer;
  COMMON_channels_t *cc = &eNB->common_channels[CC_id];
  uint8_t *rar = (uint8_t *)(dlsch_buffer+1);
  int i;
  uint8_t nb,rballoc,reps;
  uint8_t mcs,TPC,ULdelay,cqireq;
  int input_buffer_length;


  AssertFatal(RA_template != NULL, "RA is null \n");

  // subheader fixed
  rarh->E                     = 0; // First and last RAR
  rarh->T                     = 1; // 0 for E/T/R/R/BI subheader, 1 for E/T/RAPID subheader
  rarh->RAPID                 = RA_template->preamble_index; // Respond to Preamble 0 only for the moment
  RA_template->timing_offset /= 16; //T_A = N_TA/16, where N_TA should be on a 30.72Msps
  rar[0] = (uint8_t)(RA_template->timing_offset>>(2+4)); // 7 MSBs of timing advance + divide by 4
  rar[1] = (uint8_t)(RA_template->timing_offset<<(4-2))&0xf0; // 4 LSBs of timing advance + divide by 4

  int N_NB_index;

  AssertFatal(1==0,"RAR for BL/CE Still to be finished ...\n"); 

  // Copy the Msg2 narrowband
  RA_template->msg34_narrowband = RA_template->msg2_narrowband; 

  if (ce_level<2) { //CE Level 0,1, CEmodeA
    input_buffer_length =6;

    N_NB_index = get_numnarrowbandbits(cc->mib->message.dl_Bandwidth);

    rar[4] = (uint8_t)(RA_template->rnti>>8);
    rar[5] = (uint8_t)(RA_template->rnti&0xff);
    //cc->RA_template[ra_idx].timing_offset = 0;
    nb      = 0;
    rballoc = mac_computeRIV(6,1+ce_level,1); // one PRB only for UL Grant in position 1+ce_level within Narrowband
    rar[1] |= (rballoc&15)<<(4-N_NB_index); // Hopping = 0 (bit 3), 3 MSBs of rballoc

    reps    = 4;
    mcs     = 7;
    TPC     = 3; // no power increase
    ULdelay = 0;
    cqireq = 0;
    rar[2] |= ((mcs&0x8)>>3);  // mcs 10
    rar[3] = (((mcs&0x7)<<5)) | ((TPC&7)<<2) | ((ULdelay&1)<<1) | (cqireq&1);
  }
  else { // CE level 2,3 => CEModeB

    input_buffer_length =5;

    rar[3] = (uint8_t)(RA_template->rnti>>8);
    rar[4] = (uint8_t)(RA_template->rnti&0xff);
  }
  LOG_D(MAC,"[RAPROC] Frame %d Generating RAR BR (%02x|%02x.%02x.%02x.%02x.%02x.%02x) for ce_level %d, CRNTI %x,preamble %d/%d,TIMING OFFSET %d\n",
        frameP,
        *(uint8_t*)rarh,rar[0],rar[1],rar[2],rar[3],rar[4],rar[5],
        ce_level,
        RA_template->rnti,
        rarh->RAPID,RA_template->preamble_index,
        RA_template->timing_offset);

  if (opt_enabled) {
    trace_pdu(1, dlsch_buffer, input_buffer_length, eNB->Mod_id, 2, 1,
        eNB->frame, eNB->subframe, 0, 0);
    LOG_D(OPT,"[RAPROC] RAR Frame %d trace pdu for rnti %x and  rapid %d size %d\n",
          frameP, RA_template->rnti,
          rarh->RAPID, input_buffer_length);
  }

  return(RA_template->rnti);
}
#endif

//------------------------------------------------------------------------------
uint16_t
ue_process_rar(
  const module_id_t module_idP,
  const int CC_id,
  const frame_t frameP,
  const rnti_t ra_rnti,
  uint8_t* const dlsch_buffer,
  rnti_t* const t_crnti,
  const uint8_t preamble_index,
  uint8_t* selected_rar_buffer // output argument for storing the selected RAR header and RAR payload
)
//------------------------------------------------------------------------------
{
	uint16_t ret = 0; // return value

  RA_HEADER_RAPID *rarh = (RA_HEADER_RAPID *)dlsch_buffer;
  //  RAR_PDU *rar = (RAR_PDU *)(dlsch_buffer+1);
  uint8_t *rar = (uint8_t *)(dlsch_buffer+1);

        // get the last RAR payload for working with CMW500
	uint8_t n_rarpy = 0; // number of RAR payloads
	uint8_t n_rarh = 0; // number of MAC RAR subheaders
	uint8_t best_rx_rapid = -1; // the closest RAPID receive from all RARs
	while (1) {
		n_rarh++;
		if (rarh->T == 1) {
			n_rarpy++;
			LOG_D(MAC, "RAPID %d\n", rarh->RAPID);
		}

		if (rarh->RAPID == preamble_index) {
			LOG_D(PHY, "Found RAR with the intended RAPID %d\n", rarh->RAPID);
			rar = (uint8_t *)(dlsch_buffer+n_rarh + (n_rarpy-1)*6);
			break;
		}

		if (abs((int)rarh->RAPID - (int)preamble_index) < abs((int)best_rx_rapid - (int)preamble_index)) {
			best_rx_rapid = rarh->RAPID;
			rar = (uint8_t *)(dlsch_buffer+n_rarh + (n_rarpy-1)*6);
		}

		if (rarh->E == 0) {
			LOG_I(PHY, "No RAR found with the intended RAPID. The closest RAPID in all RARs is %d\n", best_rx_rapid);
			break;
		} else {
			rarh++;
		}
	};
	LOG_D(MAC, "number of RAR subheader %d; number of RAR pyloads %d\n", n_rarh, n_rarpy);

  if (CC_id>0) {
    LOG_W(MAC,"Should not have received RAR on secondary CCs! \n");
    return(0xffff);
  }

  LOG_I(MAC,"[eNB %d][RAPROC] Frame %d Received RAR (%02x|%02x.%02x.%02x.%02x.%02x.%02x) for preamble %d/%d\n",module_idP,frameP,
        *(uint8_t*)rarh,rar[0],rar[1],rar[2],rar[3],rar[4],rar[5],
        rarh->RAPID,preamble_index);
#ifdef DEBUG_RAR
  LOG_D(MAC,"[UE %d][RAPROC] rarh->E %d\n",module_idP,rarh->E);
  LOG_D(MAC,"[UE %d][RAPROC] rarh->T %d\n",module_idP,rarh->T);
  LOG_D(MAC,"[UE %d][RAPROC] rarh->RAPID %d\n",module_idP,rarh->RAPID);

  //  LOG_I(MAC,"[UE %d][RAPROC] rar->R %d\n",module_idP,rar->R);
  LOG_D(MAC,"[UE %d][RAPROC] rar->Timing_Advance_Command %d\n",module_idP,(((uint16_t)(rar[0]&0x7f))<<4) + (rar[1]>>4));
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->hopping_flag %d\n",module_idP,rar->hopping_flag);
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->rb_alloc %d\n",module_idP,rar->rb_alloc);
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->mcs %d\n",module_idP,rar->mcs);
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->TPC %d\n",module_idP,rar->TPC);
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->UL_delay %d\n",module_idP,rar->UL_delay);
  //  LOG_I(MAC,"[UE %d][RAPROC] rar->cqi_req %d\n",module_idP,rar->cqi_req);
  LOG_D(MAC,"[UE %d][RAPROC] rar->t_crnti %x\n",module_idP,(uint16_t)rar[5]+(rar[4]<<8));
#endif

  if (opt_enabled) {
    LOG_D(OPT,"[UE %d][RAPROC] CC_id %d RAR Frame %d trace pdu for ra-RNTI %x\n",
          module_idP, CC_id, frameP, ra_rnti);
    trace_pdu(1, (uint8_t*)dlsch_buffer, n_rarh + n_rarpy*6, module_idP, 2, ra_rnti,
        UE_mac_inst[module_idP].rxFrame, UE_mac_inst[module_idP].rxSubframe, 0, 0);
  }

  if (preamble_index == rarh->RAPID) {
    *t_crnti = (uint16_t)rar[5]+(rar[4]<<8);//rar->t_crnti;
    UE_mac_inst[module_idP].crnti = *t_crnti;//rar->t_crnti;
    //return(rar->Timing_Advance_Command);
    ret = ((((uint16_t)(rar[0]&0x7f))<<4) + (rar[1]>>4));
  } else {
    UE_mac_inst[module_idP].crnti=0;
    ret = (0xffff);
  }

  // move the selected RAR to the front of the RA_PDSCH buffer
  memcpy(selected_rar_buffer+0, (uint8_t*)rarh, 1);
  memcpy(selected_rar_buffer+1, (uint8_t*)rar , 6);

  return ret;

}
