/*
 * Licensed to Open Source Radio Access Network(OS-RAN) Alliance and OAI
 * Software Alliance under one or more contributor license agreements. The
 * initial OpenXG series projects are derided from OAI projects, the files from
 * OAI projects are all in compliance with OAI Public License, Version 1.1.
 * codes and files developed from OpenXG projects and from OS-RAN Alliance
 * are all under OS-RAN licenses; you may not use this file except in compliance
 * with the license.  You may get a copy of the license at:
 *	http://www.openxg.org.cn/?falu_69.html
 * For more information about OpenXG, please contact:
 * contact@openxg.org.cn
 */


#ifndef FILE_ASN1_CONVERSIONS_SEEN
#define FILE_ASN1_CONVERSIONS_SEEN

#include "BIT_STRING.h"
#include "assertions.h"

//-----------------------begin func -------------------

/*! \fn uint8_t BIT_STRING_to_uint8(BIT_STRING_t *)
 *\brief  This function extract at most a 8 bits value from a BIT_STRING_t
 *object, the exact bits number depend on the BIT_STRING_t contents. \param[in]
 *pointer to the BIT_STRING_t object. \return the extracted value.
 */
static inline uint8_t BIT_STRING_to_uint8(BIT_STRING_t* asn) {
  DevCheck((asn->size == 1), asn->size, 0, 0);

  return asn->buf[0] >> asn->bits_unused;
}

/*! \fn uint16_t BIT_STRING_to_uint16(BIT_STRING_t *)
 *\brief  This function extract at most a 16 bits value from a BIT_STRING_t
 *object, the exact bits number depend on the BIT_STRING_t contents. \param[in]
 *pointer to the BIT_STRING_t object. \return the extracted value.
 */
static inline uint16_t BIT_STRING_to_uint16(BIT_STRING_t* asn) {
  uint16_t result = 0;
  int index       = 0;

  DevCheck((asn->size > 0) && (asn->size <= 2), asn->size, 0, 0);

  switch (asn->size) {
    case 2:
      result |= asn->buf[index++] << (8 - asn->bits_unused);

    case 1:
      result |= asn->buf[index] >> asn->bits_unused;
      break;

    default:
      break;
  }

  return result;
}

/*! \fn uint32_t BIT_STRING_to_uint32(BIT_STRING_t *)
 *\brief  This function extract at most a 32 bits value from a BIT_STRING_t
 *object, the exact bits number depend on the BIT_STRING_t contents. \param[in]
 *pointer to the BIT_STRING_t object. \return the extracted value.
 */
static inline uint32_t BIT_STRING_to_uint32(BIT_STRING_t* asn) {
  uint32_t result = 0;
  int index;
  int shift;

  DevCheck((asn->size > 0) && (asn->size <= 4), asn->size, 0, 0);

  shift = ((asn->size - 1) * 8) - asn->bits_unused;
  for (index = 0; index < (asn->size - 1); index++) {
    result |= asn->buf[index] << shift;
    shift -= 8;
  }

  result |= asn->buf[index] >> asn->bits_unused;

  return result;
}

/*! \fn uint64_t BIT_STRING_to_uint64(BIT_STRING_t *)
 *\brief  This function extract at most a 64 bits value from a BIT_STRING_t
 *object, the exact bits number depend on the BIT_STRING_t contents. \param[in]
 *pointer to the BIT_STRING_t object. \return the extracted value.
 */
static inline uint64_t BIT_STRING_to_uint64(BIT_STRING_t* asn) {
  uint64_t result = 0;
  int index;
  int shift;

  DevCheck((asn->size > 0) && (asn->size <= 8), asn->size, 0, 0);

  shift = ((asn->size - 1) * 8) - asn->bits_unused;
  for (index = 0; index < (asn->size - 1); index++) {
    result |= asn->buf[index] << shift;
    shift -= 8;
  }

  result |= asn->buf[index] >> asn->bits_unused;

  return result;
}

#endif /* FILE_ASN1_CONVERSIONS_SEEN */
