/*
 * Licensed to the OpenAirInterface (OAI) Software Alliance under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The OpenAirInterface Software Alliance licenses this file to You under
 * the OAI Public License, Version 1.1  (the "License"); you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.openairinterface.org/?page_id=698
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *-------------------------------------------------------------------------------
 * For more information about the OpenAirInterface (OAI) Software Alliance:
 *      contact@openairinterface.org
 */

/*! \file common/utils/lte/prach_utils.c
 * \brief utils for PRACH  common to both PHY and MAC
 * \author R. Knopp
 * \date 2020
 * \version 0.1
 * \company Eurecom
 * \email: knopp@eurecom.fr
 * \note
 * \warning
 */

#include <stdint.h>


#include "PHY/LTE_TRANSPORT/transport_common.h"

// This is table 5.7.1-4 from 36.211
const PRACH_TDD_PREAMBLE_MAP tdd_preamble_map[64][7] = {
    // TDD Configuration Index 0
    {{1, {{0, 1, 0, 2}}},
     {1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {1, {{0, 1, 0, 2}}},
     {1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {1, {{0, 1, 0, 2}}}},
    // TDD Configuration Index 1
    {{1, {{0, 2, 0, 2}}},
     {1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {1, {{0, 2, 0, 2}}},
     {1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {1, {{0, 2, 0, 2}}}},
    // TDD Configuration Index 2
    {{1, {{0, 1, 1, 2}}},
     {1, {{0, 1, 1, 1}}},
     {1, {{0, 1, 1, 0}}},
     {1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 1, 1}}}},
    // TDD Configuration Index 3
    {{1, {{0, 0, 0, 2}}},
     {1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 2}}},
     {1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 2}}}},
    // TDD Configuration Index 4
    {{1, {{0, 0, 1, 2}}},
     {1, {{0, 0, 1, 1}}},
     {1, {{0, 0, 1, 0}}},
     {1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 1, 1}}}},
    // TDD Configuration Index 5
    {{1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 1}}}},
    // TDD Configuration Index 6
    {{2, {{0, 0, 0, 2}, {0, 0, 1, 2}}},
     {2, {{0, 0, 0, 1}, {0, 0, 1, 1}}},
     {2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {2, {{0, 0, 0, 1}, {0, 0, 0, 2}}},
     {2, {{0, 0, 0, 0}, {0, 0, 0, 1}}},
     {2, {{0, 0, 0, 0}, {1, 0, 0, 0}}},
     {2, {{0, 0, 0, 2}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 7
    {{2, {{0, 0, 0, 1}, {0, 0, 1, 1}}},
     {2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 0}, {0, 0, 0, 2}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 1}, {0, 0, 1, 0}}}},
    // TDD Configuration Index 8
    {{2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 0}, {0, 0, 0, 1}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 0}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 9
    {{3, {{0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 2}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 1}}},
     {3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 1}}},
     {3, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 10
    {{3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {0, 0, 1, 1}}},
     {3, {{0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}}},
     {3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 2}, {0, 0, 1, 0}}}},
    // TDD Configuration Index 11
    {{0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 12
    {{4, {{0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 1}, {0, 0, 1, 2}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}}},
     {4, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 2}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 0}, {1, 0, 0, 1}}},
     {4, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}}},
     {4, {{0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 13
    {{4, {{0, 0, 0, 0}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 2}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 1}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 14
    {{4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 15
    {{5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 1}, {0, 0, 1, 2}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 1}, {1, 0, 0, 2}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 0}, {1, 0, 0, 1}, {2, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}}}},
    // TDD Configuration Index 16
    {{5, {{0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}, {0, 0, 1, 2}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 1, 1}}},
     {5, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 1, 0}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 0}, {1, 0, 0, 2}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 0}, {1, 0, 0, 1}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}}},
    // TDD Configuration Index 17
    {{5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 2}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {5, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 0}, {1, 0, 0, 1}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}}},
    // TDD Configuration Index 18
    {{6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}, {0, 0, 1, 2}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}}},
     {6, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}, {2, 0, 1, 0}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {1, 0, 0, 0}, {1, 0, 0, 1}, {1, 0, 0, 2}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {1, 0, 0, 0}, {1, 0, 0, 1}, {2, 0, 0, 0}, {2, 0, 0, 1}}},
     {6, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}, {5, 0, 0, 0}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 0, 2}}}},
    // TDD Configuration Index 19
    {{0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 0, 0}, {1, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 1, 0}, {0, 0, 1, 1}, {1, 0, 1, 1}}}},
    // TDD Configuration Index 20
    {{1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 1}}}},
    // TDD Configuration Index 21
    {{1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 1}}}},

    // TDD Configuration Index 22
    {{1, {{0, 1, 1, 1}}},
     {1, {{0, 1, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 1, 0}}}},

    // TDD Configuration Index 23
    {{1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 1}}}},

    // TDD Configuration Index 24
    {{1, {{0, 0, 1, 1}}},
     {1, {{0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 1, 0}}}},

    // TDD Configuration Index 25
    {{2, {{0, 0, 0, 1}, {0, 0, 1, 1}}},
     {2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 1}, {1, 0, 0, 1}}},
     {2, {{0, 0, 0, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 1}, {0, 0, 1, 0}}}},

    // TDD Configuration Index 26
    {{3, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}}},
     {3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}}},
     {3, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}}}},

    // TDD Configuration Index 27
    {{4, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}}},
     {4, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}}},
     {4, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}}}},

    // TDD Configuration Index 28
    {{5, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}, {2, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {5, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}, {4, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {5, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}, {2, 0, 0, 1}}}},

    // TDD Configuration Index 29
    {{6, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}, {2, 0, 0, 1}, {2, 0, 1, 1}}},
     {6, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}, {2, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}, {4, 0, 0, 1}, {5, 0, 0, 1}}},
     {6, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}, {5, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}, {2, 0, 0, 1}, {2, 0, 1, 0}}}},

    // TDD Configuration Index 30
    {{1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 1}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 1}}}},

    // TDD Configuration Index 31
    {{1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 1}}},
     {1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 1}}}},

    // TDD Configuration Index 32
    {{1, {{0, 1, 1, 1}}},
     {1, {{0, 1, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 1, 0}}}},

    // TDD Configuration Index 33
    {{1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 1}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 1}}}},

    // TDD Configuration Index 34
    {{1, {{0, 0, 1, 1}}},
     {1, {{0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 1, 0}}}},

    // TDD Configuration Index 35
    {{2, {{0, 0, 0, 1}, {0, 0, 1, 1}}},
     {2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 1}, {1, 0, 0, 1}}},
     {2, {{0, 0, 0, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 1}, {0, 0, 1, 0}}}},

    // TDD Configuration Index 36
    {{3, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}}},
     {3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}}},
     {3, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}}}},

    // TDD Configuration Index 37
    {{4, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}}},
     {4, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}}},
     {4, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}}}},

    // TDD Configuration Index 38
    {{5, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}, {2, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {5, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}, {4, 0, 0, 1}}},
     {5, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {5, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}, {2, 0, 0, 1}}}},

    // TDD Configuration Index 39
    {{6, {{0, 0, 0, 1}, {0, 0, 1, 1}, {1, 0, 0, 1}, {1, 0, 1, 1}, {2, 0, 0, 1}, {2, 0, 1, 1}}},
     {6, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}, {2, 0, 0, 0}, {2, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 1}, {1, 0, 0, 1}, {2, 0, 0, 1}, {3, 0, 0, 1}, {4, 0, 0, 1}, {5, 0, 0, 1}}},
     {6, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}, {4, 0, 0, 0}, {5, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {6, {{0, 0, 0, 1}, {0, 0, 1, 0}, {1, 0, 0, 1}, {1, 0, 1, 0}, {2, 0, 0, 1}, {2, 0, 1, 0}}}},

    // TDD Configuration Index 40
    {{1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 1, 0, 0}}}},
    // TDD Configuration Index 41
    {{1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 2, 0, 0}}}},

    // TDD Configuration Index 42
    {{1, {{0, 1, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}}},

    // TDD Configuration Index 43
    {{1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {1, {{0, 0, 0, 0}}}},

    // TDD Configuration Index 44
    {{1, {{0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}}}},

    // TDD Configuration Index 45
    {{2, {{0, 0, 0, 0}, {0, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
     {2, {{0, 0, 0, 0}, {1, 0, 0, 0}}}},

    // TDD Configuration Index 46
    {{3, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {3, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}}}},

    // TDD Configuration Index 47
    {{4, {{0, 0, 0, 0}, {0, 0, 1, 0}, {1, 0, 0, 0}, {1, 0, 1, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {0, {{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}},
     {4, {{0, 0, 0, 0}, {1, 0, 0, 0}, {2, 0, 0, 0}, {3, 0, 0, 0}}}}};

uint8_t get_prach_fmt(int prach_ConfigIndex,int frame_type) {
  if (frame_type == FDD) // FDD
    return(prach_ConfigIndex>>4);
  else {
    if (prach_ConfigIndex < 20)
      return (0);

    if (prach_ConfigIndex < 30)
      return (1);

    if (prach_ConfigIndex < 40)
      return (2);

    if (prach_ConfigIndex < 48)
      return (3);
    else
      return (4);
  }
}

uint8_t get_prach_prb_offset(int frame_type,
			     int tdd_config,
                             int N_RB_UL,
                             uint8_t prach_ConfigIndex,
                             uint8_t n_ra_prboffset,
                             uint8_t tdd_mapindex, uint16_t Nf) {
  uint8_t n_ra_prb;
  uint8_t f_ra,t1_ra;
  uint8_t prach_fmt = get_prach_fmt(prach_ConfigIndex,frame_type);
  uint8_t Nsp=2;

  if (frame_type == TDD) { // TDD
    if (tdd_preamble_map[prach_ConfigIndex][tdd_config].num_prach==0) {
      LOG_E(PHY, "Illegal prach_ConfigIndex %"PRIu8"", prach_ConfigIndex);
      return(-1);
    }

    // adjust n_ra_prboffset for frequency multiplexing (p.36 36.211)
    f_ra = tdd_preamble_map[prach_ConfigIndex][tdd_config].map[tdd_mapindex].f_ra;

    if (prach_fmt < 4) {
      if ((f_ra&1) == 0) {
        n_ra_prb = n_ra_prboffset + 6*(f_ra>>1);
      } else {
        n_ra_prb = N_RB_UL - 6 - n_ra_prboffset + 6*(f_ra>>1);
      }
    } else {
      if ((tdd_config >2) && (tdd_config<6))
        Nsp = 2;

      t1_ra = tdd_preamble_map[prach_ConfigIndex][tdd_config].map[0].t1_ra;

      if ((((Nf&1)*(2-Nsp)+t1_ra)&1) == 0) {
        n_ra_prb = 6*f_ra;
      } else {
        n_ra_prb = N_RB_UL - 6*(f_ra+1);
      }
    }
  } else { //FDD
    n_ra_prb = n_ra_prboffset;
  }

  return(n_ra_prb);
}

int is_prach_subframe0(int tdd_config,int frame_type,uint8_t prach_ConfigIndex,uint32_t frame, uint8_t subframe) {
  uint8_t t0_ra;
  uint8_t t1_ra;
  uint8_t t2_ra;
  int prach_mask = 0;

  if (frame_type == FDD) { //FDD
    //implement Table 5.7.1-2 from 36.211 (Rel-10, p.41)
    if ((((frame&1) == 1) && (subframe < 9)) ||
        (((frame&1) == 0) && (subframe == 9)))  // This is an odd frame, ignore even-only PRACH frames
      if (((prach_ConfigIndex&0xf)<3) || // 0,1,2,16,17,18,32,33,34,48,49,50
          ((prach_ConfigIndex&0x1f)==18) || // 18,50
          ((prach_ConfigIndex&0xf)==15))   // 15,47
        return(0);

    switch (prach_ConfigIndex&0x1f) {
      case 0:
      case 3:
        if (subframe==1) prach_mask = 1;

        break;

      case 1:
      case 4:
        if (subframe==4) prach_mask = 1;

        break;

      case 2:
      case 5:
        if (subframe==7) prach_mask = 1;

        break;

      case 6:
        if ((subframe==1) || (subframe==6)) prach_mask=1;

        break;

      case 7:
        if ((subframe==2) || (subframe==7)) prach_mask=1;

        break;

      case 8:
        if ((subframe==3) || (subframe==8)) prach_mask=1;

        break;

      case 9:
        if ((subframe==1) || (subframe==4) || (subframe==7)) prach_mask=1;

        break;

      case 10:
        if ((subframe==2) || (subframe==5) || (subframe==8)) prach_mask=1;

        break;

      case 11:
        if ((subframe==3) || (subframe==6) || (subframe==9)) prach_mask=1;

        break;

      case 12:
        if ((subframe&1)==0) prach_mask=1;

        break;

      case 13:
        if ((subframe&1)==1) prach_mask=1;

        break;

      case 14:
        prach_mask=1;
        break;

      case 15:
        if (subframe==9) prach_mask=1;

        break;
    }
  } else { // TDD
    AssertFatal(prach_ConfigIndex<64,
                "Illegal prach_ConfigIndex %d for ",prach_ConfigIndex);
    AssertFatal(tdd_preamble_map[prach_ConfigIndex][tdd_config].num_prach>0,
                "Illegal prach_ConfigIndex %d for ",prach_ConfigIndex);
    t0_ra = tdd_preamble_map[prach_ConfigIndex][tdd_config].map[0].t0_ra;
    t1_ra = tdd_preamble_map[prach_ConfigIndex][tdd_config].map[0].t1_ra;
    t2_ra = tdd_preamble_map[prach_ConfigIndex][tdd_config].map[0].t2_ra;
#ifdef PRACH_DEBUG
    LOG_I(PHY,"[PRACH] Checking for PRACH format (ConfigIndex %d) in TDD subframe %d (%d,%d,%d)\n",
          prach_ConfigIndex,
          subframe,
          t0_ra,t1_ra,t2_ra);
#endif

    if ((((t0_ra == 1) && ((frame &1)==0))||  // frame is even and PRACH is in even frames
         ((t0_ra == 2) && ((frame &1)==1))||  // frame is odd and PRACH is in odd frames
         (t0_ra == 0)) &&                                // PRACH is in all frames
        (((subframe<5)&&(t1_ra==0)) ||                   // PRACH is in 1st half-frame
         (((subframe>4)&&(t1_ra==1))))) {                // PRACH is in 2nd half-frame
      if ((prach_ConfigIndex<48) &&                          // PRACH only in normal UL subframe
          (((subframe%5)-2)==t2_ra)) prach_mask=1;
      else if ((prach_ConfigIndex>47) && (((subframe%5)-1)==t2_ra)) prach_mask=1;      // PRACH can be in UpPTS
    }
  }

  return(prach_mask);
}

