#!/bin/sh

#
# This script is designed to quickly create lots of files in underlying
# test-* directories, do lots of other magic stuff and exit cleanly.
#

set -e

if [ "x$1" = "x" ]; then
	echo "Usage: $0 check-src/<check-NN.c>"
	exit 1
fi

srcdir="${srcdir:-.}"
abs_top_srcdir="${abs_top_srcdir:-$(pwd)/../../}"
abs_top_builddir="${abs_top_builddir:-$(pwd)/../../}"

if echo "$*" | grep -q -- -- ; then
    TEST_DRIVER=$(echo "$*"  | sed -e 's/ -- .*/ -- /g')
    source_full=$(echo "$*"  | sed -e 's/.* //g')
else
    TEST_DRIVER=""
    source_full="$1"
fi

# Compute the .asn1 spec name by the given file name.
source_short=$(echo "$source_full" | sed -e 's/.*\///')
testno=$(echo "$source_short" | cut -f2 -d'-' | cut -f1 -d'.')

args=$(echo "$source_short" | sed -E -e 's/\.c+$//')
source_obj=$(echo "$source_short" | sed -E -e 's/\.c+$/.o/')
ext=$(echo "$source_short" | sed -E -e 's/.*\.(c+)$/\1/')

OFS=$IFS
IFS="."
set $args
data_dir=$(echo "$1" | sed -e s/check-/data-/)
shift
IFS=$OFS
AFLAGS="$*"

if [ -d ${data_dir} ]; then
    OPT_DATA_DIR="../${data_dir}"
else
    OPT_DATA_DIR=""
fi

if test "${LIBFUZZER_CFLAGS}" && grep LLVMFuzzer ${source_full} > /dev/null;
then
    MAKE_FUZZER=yes
else
    MAKE_FUZZER=no
fi

# Assume the test fails. Will be removed when it passes well.
testdir=test-${args}
if [ -f "${testdir}-FAILED" ]; then
    rm -rf "${testdir}"
fi
touch "${testdir}-FAILED"

mkdir -p "${testdir}"
ln -fns "../${source_full}" "${testdir}"

asn_module=$(echo "${abs_top_srcdir}/tests/tests-asn1c-compiler/${testno}"-*.asn1)

AUTOGENERATED="# This file is autogenerated by $0 ${source_full} ${AFLAGS}"

# Create a common Makefile for the project
cat <<END_MAKEFILE > "${testdir}/Makefile"
${AUTOGENERATED}
COMMON_FLAGS= -I. -I${abs_top_builddir}
CFLAGS = \${COMMON_FLAGS} ${CFLAGS:-} ${DEFS} -g -O1
CFLAGS += -DSRCDIR=../${srcdir}
CXXFLAGS = \${CFLAGS} ${CXXFLAGS}
LIBFUZZER_CFLAGS = ${LIBFUZZER_CFLAGS}
LDFLAGS = ${LDFLAGS:-}
ASAN_ENV_FLAGS = ${ASAN_ENV_FLAGS:-}

ASN_PROGRAM = check-program
ASN_PROGRAM_SRCS = ${source_short}

CC ?= ${CC}
CXX ?= ${CXX}

-include converter-example.mk

all: compiled-module
	\$(MAKE) check-program

# Compile the corresponding .asn1 spec.
compiled-module: ${asn_module} ${abs_top_builddir}/asn1c/asn1c
	${abs_top_builddir}/asn1c/asn1c		\\
		-S ${abs_top_srcdir}/skeletons	\\
		-flink-skeletons 		\\
		-Wdebug-compiler		\\
		${AFLAGS} ${asn_module}
	rm -f converter-example.c
	@touch compiled-module

END_MAKEFILE


if [ "${MAKE_FUZZER}" != "yes" ]; then
    CHECK_FUZZER="@echo \"No fuzzer defined, skipping.\""
cat <<END_MAKEFILE >> "${testdir}/Makefile"
.PHONY: check-fuzzer
check-fuzzer:
END_MAKEFILE
else
    CHECK_FUZZER="ASAN_OPTIONS=detect_leaks=1 UBSAN_OPTIONS=print_stacktrace=1 ./check-fuzzer -timeout=3 -max_total_time=60 -max_len=512 ${OPT_DATA_DIR}"
cat <<END_MAKEFILE >> "${testdir}/Makefile"
check-fuzzer: ${source_short} \$(ASN_LIBRARY)
	${CC} \$(CFLAGS) \$(LIBFUZZER_CFLAGS) \$(LDFLAGS) -o check-fuzzer \$^
END_MAKEFILE
fi

cat <<END_MAKEFILE >> "${testdir}/Makefile"

check-succeeded: compiled-module \$(ASN_LIBRARY) ${source_short}
	\$(MAKE) check-program
	\$(MAKE) check-fuzzer
	@rm -f check-succeeded
	\${ASAN_ENV_FLAGS} ./check-program
	\$(MAKE) fuzz
	@touch check-succeeded
	@echo "OK: ${source_full}"

.PHONY: fuzz
fuzz:
	${CHECK_FUZZER}

check: check-succeeded

END_MAKEFILE

# Perform building and checking
${TEST_DRIVER} ${MAKE:-make} -C "${testdir}" check

# Make sure the test is not marked as failed any longer.
rm -f "${testdir}-FAILED"
